<?php
session_start();
require_once '../config/db.php';
require_once '../auth/auth.php';
require_once '../includes/functions.php';

// Check if user has permission to manage suppliers
if (!hasPermission('manage_suppliers')) {
    header('Location: dashboard.php');
    exit;
}

// Get all suppliers
$stmt = $pdo->query("SELECT id, name FROM suppliers ORDER BY name");
$suppliers = $stmt->fetchAll();
// Get all products
$stmt = $pdo->query("SELECT id, name FROM products WHERE status = 'active' ORDER BY name");
$products = $stmt->fetchAll();

$success = '';
$error = '';

if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['receive_item'])) {
    try {
        $supplier_id = $_POST['supplier_id'];
        $product_ids = $_POST['product_id'];
        $quantities = $_POST['quantity'];
        $unit_prices = $_POST['unit_price'];
        $batch_numbers = $_POST['batch_number'];
        $notes_arr = $_POST['notes'];
        $serial_numbers_arr = $_POST['serial_numbers'];
        $item_count = count($product_ids);
        if (empty($supplier_id) || $item_count === 0) {
            throw new Exception('Supplier and at least one item are required.');
        }
        $pdo->beginTransaction();
        for ($i = 0; $i < $item_count; $i++) {
            if (empty($product_ids[$i]) || empty($quantities[$i]) || empty($unit_prices[$i]) || empty($batch_numbers[$i])) {
                throw new Exception('All item fields are required.');
            }
            $stmt = $pdo->prepare("
                INSERT INTO supplier_receipts (
                    supplier_id, product_id, quantity, unit_price, batch_number, notes, created_by, created_at
                ) VALUES (?, ?, ?, ?, ?, ?, ?, NOW())
            ");
            $stmt->execute([
                $supplier_id,
                $product_ids[$i],
                $quantities[$i],
                $unit_prices[$i],
                $batch_numbers[$i],
                $notes_arr[$i],
                $_SESSION['user_id']
            ]);
            $receipt_id = $pdo->lastInsertId();

            // Add serial numbers if provided
            if (!empty($serial_numbers_arr[$i])) {
                $serial_numbers = array_map('trim', explode(',', $serial_numbers_arr[$i]));
                $serial_numbers = array_filter($serial_numbers); // Remove empty values
                if (!empty($serial_numbers)) {
                    addProductSerialNumbers($product_ids[$i], $serial_numbers, 'supplier_receipt', $receipt_id);
                }
            }
        }
        $pdo->commit();
        $success = 'Receipts created successfully!';
    } catch (Exception $e) {
        if ($pdo->inTransaction()) $pdo->rollBack();
        $error = 'Error creating receipts: ' . $e->getMessage();
    }
}
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Receive Supplier Items</title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css" rel="stylesheet">
    <link href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css" rel="stylesheet">
    <link href="https://fonts.googleapis.com/css2?family=Inter:wght@400;600;700&display=swap" rel="stylesheet">
    <style>
        body { background: linear-gradient(135deg, #f8fafc 0%, #e0e7ff 100%); font-family: 'Inter', sans-serif; }
        .card { border-radius: 16px; box-shadow: 0 2px 12px rgba(37,99,235,0.06); border: none; }
        .form-label { font-weight: 600; color: #2563eb; }
        .modal-title { color: #2563eb; font-weight: 700; }
        .table { background: #fff; border-radius: 12px; overflow: hidden; }
        .table th, .table td { vertical-align: middle; }
    </style>
</head>
<body>
<div class="container py-4">
    <div class="row justify-content-center">
        <div class="col-lg-10">
            <div class="card">
                <div class="card-header d-flex justify-content-between align-items-center">
                    <h4 class="mb-0">Create New Receipt</h4>
                </div>
                <div class="card-body">
                    <?php if ($success): ?>
                        <div class="alert alert-success"><?= $success ?></div>
                    <?php endif; ?>
                    <?php if ($error): ?>
                        <div class="alert alert-danger"><?= $error ?></div>
                    <?php endif; ?>
                    <form method="post" class="needs-validation" novalidate>
                        <div class="row g-3">
                            <div class="col-md-6">
                                <label class="form-label">Supplier *</label>
                                <select class="form-select form-select-sm" name="supplier_id" required>
                                    <option value="">Select Supplier</option>
                                    <?php foreach ($suppliers as $supplier): ?>
                                        <option value="<?php echo $supplier['id']; ?>"><?php echo htmlspecialchars($supplier['name']); ?></option>
                                    <?php endforeach; ?>
                                </select>
                            </div>
                            <div class="col-12">
                                <label class="form-label">Items *</label>
                                <table class="table table-bordered" id="itemsTable">
                                    <thead>
                                        <tr>
                                            <th>Product</th>
                                            <th>Quantity</th>
                                            <th>Unit Price</th>
                                            <th>Batch Number</th>
                                            <th>Notes</th>
                                            <th>Serial Numbers</th>
                                            <th></th>
                                        </tr>
                                    </thead>
                                    <tbody>
                                        <tr>
                                            <td>
                                                <select class="form-select form-select-sm" name="product_id[]" required>
                                                    <option value="">Select Product</option>
                                                    <?php foreach ($products as $product): ?>
                                                        <option value="<?php echo $product['id']; ?>"><?php echo htmlspecialchars($product['name']); ?></option>
                                                    <?php endforeach; ?>
                                                </select>
                                            </td>
                                            <td><input type="number" class="form-control form-control-sm" name="quantity[]" min="1" required></td>
                                            <td><input type="number" class="form-control form-control-sm" name="unit_price[]" step="0.01" min="0.01" required></td>
                                            <td><input type="text" class="form-control form-control-sm" name="batch_number[]" required></td>
                                            <td><input type="text" class="form-control form-control-sm" name="notes[]"></td>
                                            <td><input type="text" class="form-control form-control-sm" name="serial_numbers[]"></td>
                                            <td><button type="button" class="btn btn-danger btn-sm remove-item">&times;</button></td>
                                        </tr>
                                    </tbody>
                                </table>
                                <button type="button" class="btn btn-success btn-sm" id="addItemRow"><i class="fas fa-plus"></i> Add Another Item</button>
                            </div>
                        </div>
                        <div class="mt-4">
                            <button type="submit" name="receive_item" class="btn btn-primary">
                                <i class="fas fa-save"></i> Create Receipt
                            </button>
                        </div>
                    </form>
                </div>
            </div>
        </div>
    </div>
</div>
<script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/js/bootstrap.bundle.min.js"></script>
<script>
document.getElementById('addItemRow').addEventListener('click', function() {
    const table = document.getElementById('itemsTable').querySelector('tbody');
    const newRow = table.rows[0].cloneNode(true);
    newRow.querySelectorAll('input, select').forEach(input => input.value = '');
    table.appendChild(newRow);
});
document.getElementById('itemsTable').addEventListener('click', function(e) {
    if (e.target.classList.contains('remove-item')) {
        const row = e.target.closest('tr');
        if (row.parentNode.rows.length > 1) row.remove();
    }
});
// Form validation
(function () {
    'use strict'
    var forms = document.querySelectorAll('.needs-validation')
    Array.prototype.slice.call(forms).forEach(function (form) {
        form.addEventListener('submit', function (event) {
            if (!form.checkValidity()) {
                event.preventDefault()
                event.stopPropagation()
            }
            form.classList.add('was-validated')
        }, false)
    })
})()
</script>
</body>
</html> 